﻿Public Class MainForm
    Private ovXML As XDocument

    Private boldFont As Font
    Private ignoreEvents As Boolean

    Private Class Language
        Private mDisplayName As String
        Private mCode As String

        Public Sub New(code As String, displayName As String)
            mCode = code
            mDisplayName = displayName
        End Sub

        Public Sub New(code As String)
            mCode = code

            mDisplayName = "Global"
            Try
                Dim ci = New Globalization.CultureInfo(code)
                mDisplayName = ci.DisplayName
            Catch
            End Try
        End Sub

        Public ReadOnly Property Code As String
            Get
                Return mCode
            End Get
        End Property

        Public ReadOnly Property DisplayName As String
            Get
                Return mDisplayName
            End Get
        End Property

        Public Overrides Function Equals(obj As Object) As Boolean
            If TypeOf obj Is Language Then
                Dim lang = CType(obj, Language)
                Return lang.Code = mCode
            Else
                Return False
            End If
        End Function

        Public Overrides Function ToString() As String
            Return mDisplayName
        End Function
    End Class

    Private Sub MainForm_Load(sender As System.Object, e As System.EventArgs) Handles MyBase.Load
        boldFont = New Font(Me.Font.FontFamily, Me.Font.Size, FontStyle.Bold, Me.Font.Unit)

        LoadXML()
        AddEventHandlers()
    End Sub

    Private Sub LoadXML()
        Do While Not IO.File.Exists(My.Settings.xmlFileName)
            Using dlg = New OpenFileDialog()
                dlg.DefaultExt = "xml"
                dlg.FileName = My.Settings.xmlFileName
                dlg.Filter = "OnlineVideos Configuration (*.xml)|*.xml"
                dlg.Title = "Select OnlineViedeos Configuration File"

                Select Case dlg.ShowDialog(Me)
                    Case Windows.Forms.DialogResult.OK
                        My.Settings.xmlFileName = dlg.FileName
                    Case Windows.Forms.DialogResult.Cancel
                        Me.Close()
                End Select
            End Using
        Loop

        IO.File.Copy(My.Settings.xmlFileName, My.Settings.xmlFileName + "-backup", True)
        ovXML = XDocument.Load(My.Settings.xmlFileName)

        LoadLanguages()
        ApplyFilter()
    End Sub

    Private Sub LoadLanguages()
        Dim langs = (From s In ovXML...<Site> Select s.@lang).OrderBy(Function(lang) lang).Distinct()

        ignoreEvents = True

        cbLang.Items.Clear()
        cbLang.Items.Add(New Language("", "<All>"))

        For Each lang In langs
            cbLang.Items.Add(New Language(lang))
        Next
        cbLang.SelectedIndex = 0

        ignoreEvents = False
    End Sub

    Private Sub AddEventHandlers()
        AddHandler tbFilter.TextChanged, Sub() ApplyFilter()
        AddHandler cbLang.SelectedIndexChanged, Sub() ApplyFilter()

        AddHandler lvSites.ItemChecked, Sub(sender As Object, e As ItemCheckedEventArgs)
                                            If ignoreEvents Then Exit Sub

                                            Dim ovSite = CType(e.Item.Tag, XElement)
                                            ovSite.@enabled = e.Item.Checked.ToString().ToLower()
                                        End Sub

        AddHandler lvSites.AfterLabelEdit, Sub(sender As Object, e As LabelEditEventArgs)
                                               If ignoreEvents Then Exit Sub

                                               Dim ovSite = CType(lvSites.Items(e.Item).Tag, XElement)
                                               If e.Label <> "" Then ovSite.@name = e.Label
                                           End Sub

        AddHandler btnSelAll.Click, Sub()
                                        For Each item As ListViewItem In lvSites.Items
                                            item.Checked = True
                                        Next
                                    End Sub

        AddHandler btnSelInv.Click, Sub()
                                        For Each item As ListViewItem In lvSites.Items
                                            item.Checked = Not item.Checked
                                        Next
                                    End Sub

        AddHandler btnSelNon.Click, Sub()
                                        For Each item As ListViewItem In lvSites.Items
                                            item.Checked = False
                                        Next
                                    End Sub

        AddHandler cbSelectedFilter.Click, Sub()
                                               Select Case cbSelectedFilter.CheckState
                                                   Case CheckState.Checked : cbSelectedFilter.CheckState = CheckState.Indeterminate
                                                   Case CheckState.Indeterminate : cbSelectedFilter.CheckState = CheckState.Unchecked
                                                   Case CheckState.Unchecked : cbSelectedFilter.CheckState = CheckState.Checked
                                               End Select

                                               ApplyFilter()
                                           End Sub
    End Sub

    Private Sub ApplyFilter()
        If ignoreEvents Then Exit Sub
        ignoreEvents = True

        Dim sites As IEnumerable(Of XElement)

        Dim textFilter As String = tbFilter.Text
        Dim langFilter As Language = cbLang.SelectedItem

        If textFilter = "" Then
            sites = ovXML...<Site>
        Else
            textFilter = textFilter.ToLower()
            sites = (From s In ovXML...<Site>
                     Where s.@name.ToLower().Contains(textFilter) OrElse ((s.<Description>.Value IsNot Nothing) AndAlso s.<Description>.Value.ToLower().Contains(textFilter))
                     Select s).OrderBy(Function(s) s.@lang)
        End If

        If langFilter.Code <> "" Then
            sites = sites.Where(Function(s) s.@lang = langFilter.Code)
        End If

        Select Case cbSelectedFilter.CheckState
            Case CheckState.Checked : sites = sites.Where(Function(s) s.@enabled = "true")
            Case CheckState.Unchecked : sites = sites.Where(Function(s) s.@enabled = "false")
        End Select

        LoadSites(sites.OrderBy(Function(s) s.@lang))

        btnSelAll.Enabled = (sites.Count > 0)
        btnSelInv.Enabled = (sites.Count > 0)
        btnSelNon.Enabled = (sites.Count > 0)

        ignoreEvents = False
    End Sub

    Private Sub LoadSites(sites As IEnumerable(Of XElement))
        Dim cGroup As ListViewGroup = New ListViewGroup()

        lvSites.Items.Clear()

        For Each ovSite In sites
            If cGroup.Tag <> ovSite.@lang Then
                Dim lang = New Language(ovSite.@lang)

                cGroup = New ListViewGroup(lang.DisplayName)
                cGroup.Tag = lang.Code

                lvSites.Groups.Add(cGroup)
            End If

            Dim newItem = New ListViewItem(ovSite.@name)
            newItem.Group = cGroup
            newItem.SubItems.Add(ovSite.<Description>.Value)
            newItem.Checked = Boolean.Parse(ovSite.@enabled)
            newItem.Tag = ovSite

            newItem.Font = boldFont
            newItem.UseItemStyleForSubItems = False

            lvSites.Items.Add(newItem)
        Next

        lvSites.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent)
    End Sub

    Private Sub btnCancel_Click(sender As System.Object, e As System.EventArgs) Handles btnCancel.Click
        Me.Close()
    End Sub

    Private Sub btnSave_Click(sender As System.Object, e As System.EventArgs) Handles btnSave.Click
        ovXML.Save(My.Settings.xmlFileName)
        Me.Close()
    End Sub

    Private Sub btnRevert_Click(sender As System.Object, e As System.EventArgs) Handles btnRevert.Click
        If MsgBox("Are you sure you want to reload the OnlineVideos configuration and lose all your changes?", MsgBoxStyle.Question Or MsgBoxStyle.YesNo) = MsgBoxResult.Yes Then
            LoadXML()
        End If
    End Sub
End Class